/************************************************************************************
* Implements PHY-PIB primitives
*
* (c) Copyright 2009, Freescale, Inc.  All rights reserved.
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale Semiconductor.
*
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Includes
*************************************************************************************
************************************************************************************/

#include "EmbeddedTypes.h"
#include "PortConfig.h"
#include "TransceiverDrv.h"
#include "Phy.h"
#include "PhyPib.h"
#include "PhyPibExtended.h"

#ifndef gFreqBand_902__928MHz_d
//#define gFreqBand_902__928MHz_d 1	                                        //Defined in the preprocessor lines Compiler->Preprocessor options
#define gFreqBand_902__928MHz_d 0
#endif

#ifndef gFreqBand_950__958MHz_d
#define gFreqBand_950__958MHz_d 0
#endif

#ifndef gFreqBand_863__870MHz_d
#define gFreqBand_863__870MHz_d 0
#endif



/************************************************************************************
*************************************************************************************
* Private memory declarations
*************************************************************************************
************************************************************************************/
uint8_t mFreqBand;


/************************************************************************************
*************************************************************************************
* Public memory declarations
*************************************************************************************
************************************************************************************/

phyRFConstants_t const phyPibRFConstants[] =
{
#if (gFreqBand_863__870MHz_d == 1)
// 863 MHz, Operating Mode 1
  {
  //channel center frequency 0
    863125000,
  //channel spacing
    200000,
  //total number of channels
    34,
  //Radio bit rate register
    Bitrate_50000,
  //Radio frequency deviation register
    Fdev_25000,
  //number of RSSI measurements
    4,//TODO
  //Radio RxBw register
    (DccFreq_7 | RxBw_83300),
  //Radio RxAfcBw register
    (DccFreq_7 | RxBw_83300),
  //Radio modulation parameters
    (DataModul_Modulation_Fsk | DataModul_ModulationShaping_BT_05),
  //CCA threshold
  //Must be 10 dB above receiver sensitivity multiplied by 2
  //Where receiver sensitivity S = S0 + 10log10(R/R0), R0 = 50kbps, R is the bit rate in kbps
    (160 >> 1)
  },
// 863 MHz, Operating Mode 2
  {
  //channel center frequency 0
    863225000,
  //channel spacing
    400000,
  //total number of channels
    17,
  //Radio bit rate register
    Bitrate_100000,
  //Radio frequency deviation register
    Fdev_50000,
  //number of RSSI measurements
    4,//TODO
  //Radio RxBw register
    (DccFreq_7 | RxBw_166700),
  //Radio RxAfcBw register
    (DccFreq_7 | RxBw_166700),
  //Radio modulation parameters
    (DataModul_Modulation_Fsk | DataModul_ModulationShaping_BT_05),
  //CCA threshold
  //Must be 10 dB above receiver sensitivity multiplied by 2
  //Where receiver sensitivity S = S0 + 10log10(R/R0), R0 = 50kbps, R is the bit rate in kbps
    (167 >> 1)
  },
#endif
#if (gFreqBand_902__928MHz_d == 1)

#if 0
//915 MHz, Operating Mode 1
  {
  //channel center frequency 0
    902200000,
  //channel spacing
    200000,
  //total number of channels
    129,
  //Radio bit rate register
    Bitrate_50000,
  //Radio frequency deviation register
    Fdev_25000,
  //number of RSSI measurements
    4,//TODO
  //Radio RxBw register
    (DccFreq_7 | RxBw_83300),
  //Radio RxAfcBw register
    (DccFreq_7 | RxBw_83300),
  //Radio modulation parameters
    (DataModul_Modulation_Fsk | DataModul_ModulationShaping_BT_05),
  //CCA threshold
  //Must be 10 dB above receiver sensitivity multiplied by 2
  //Where receiver sensitivity S = S0 + 10log10(R/R0), R0 = 50kbps, R is the bit rate in kbps
    (160 >> 1)
  },
//915 MHz, Operating Mode 3
  {
  //channel center frequency 0
    902400000,
  //channel spacing
    400000,
  //total number of channels
    64,
  //Radio bit rate register
    Bitrate_200000,
  //Radio frequency deviation register
    Fdev_50049,
  //number of RSSI measurements
    4,//TODO
  //Radio RxBw register
    (DccFreq_7 | RxBw_250000),
  //Radio RxAfcBw register
    (DccFreq_7 | RxBw_250000),
  //Radio modulation parameters
    (DataModul_Modulation_Fsk | DataModul_ModulationShaping_BT_05),
  //CCA threshold
  //Must be 10 dB above receiver sensitivity multiplied by 2
  //Where receiver sensitivity S = S0 + 10log10(R/R0), R0 = 50kbps, R is the bit rate in kbps
    (173 >> 1)
  },
#endif

//915 MHz, Operating Mode 1
  {
  //channel center frequency 0
    902500000,
  //channel spacing
    1000000,
  //total number of channels
    25,
  //Radio bit rate register
    Bitrate_50000,
  //Radio frequency deviation register
    Fdev_170000,
  //number of RSSI measurements
    4,//TODO
  //Radio RxBw register
    (DccFreq_2 | RxBw_250000),
  //Radio RxAfcBw register
    (DccFreq_2 | RxBw_250000),
  //Radio modulation parameters
    (DataModul_Modulation_Fsk | DataModul_ModulationShaping_BT_1),
  //CCA threshold
  //Must be 10 dB above receiver sensitivity multiplied by 2
  //Where receiver sensitivity S = S0 + 10log10(R/R0), R0 = 50kbps, R is the bit rate in kbps
    (160 >> 1)
  },
//915 MHz, Operating Mode 2
  {
  //channel center frequency 0
    902200000,
  //channel spacing
    2000000,
  //total number of channels
    12,
  //Radio bit rate register
    Bitrate_150000,
  //Radio frequency deviation register
    Fdev_170000,
  //number of RSSI measurements
    4,//TODO
  //Radio RxBw register
    (DccFreq_2 | RxBw_250000),
  //Radio RxAfcBw register
    (DccFreq_2 | RxBw_250000),
  //Radio modulation parameters
    (DataModul_Modulation_Fsk | DataModul_ModulationShaping_BT_05),
  //CCA threshold
  //Must be 10 dB above receiver sensitivity multiplied by 2
  //Where receiver sensitivity S = S0 + 10log10(R/R0), R0 = 50kbps, R is the bit rate in kbps
    (160 >> 1)
  },
//915 MHz, Operating Mode 3
  {
  //channel center frequency 0
    902400000,
  //channel spacing
    2000000,
  //total number of channels
    12,
  //Radio bit rate register
    Bitrate_200000,
  //Radio frequency deviation register
    Fdev_180000,
  //number of RSSI measurements
    4,//TODO
  //Radio RxBw register
    (DccFreq_2 | RxBw_333300),
  //Radio RxAfcBw register
    (DccFreq_2 | RxBw_333300),
  //Radio modulation parameters
    (DataModul_Modulation_Fsk | DataModul_ModulationShaping_BT_05),
  //CCA threshold
  //Must be 10 dB above receiver sensitivity multiplied by 2
  //Where receiver sensitivity S = S0 + 10log10(R/R0), R0 = 50kbps, R is the bit rate in kbps
    (173 >> 1)
  },

#endif
#if (gFreqBand_950__958MHz_d == 1)
//950 MHz, Operating Mode 1
  {
  //channel center frequency 0
    951000000,
  //channel spacing
    200000,
  //total number of channels
    33,
  //Radio bit rate register
    Bitrate_50000,
  //Radio frequency deviation register
    Fdev_25000,
  //number of RSSI measurements
    4,//TODO
  //Radio RxBw register
    (DccFreq_7 | RxBw_83300),
  //Radio RxAfcBw register
    (DccFreq_7 | RxBw_83300),
  //Radio modulation parameters
    (DataModul_Modulation_Fsk | DataModul_ModulationShaping_BT_05),
  //CCA threshold
  //Must be 10 dB above receiver sensitivity multiplied by 2
  //Where receiver sensitivity S = S0 + 10log10(R/R0), R0 = 50kbps, R is the bit rate in kbps
    (160 >> 1)
  },
//950 MHz, Operating Mode 3
  {
  //channel center frequency 0
    951200000,
  //channel spacing
    600000,
  //total number of channels
    11,
  //Radio bit rate register
    Bitrate_200000,
  //Radio frequency deviation register
    Fdev_100000,
  //number of RSSI measurements
    4,//TODO
  //Radio RxBw register
    (DccFreq_7 | RxBw_333300),
  //Radio RxAfcBw register
    (DccFreq_7 | RxBw_333300),
  //Radio modulation parameters
    (DataModul_Modulation_Fsk | DataModul_ModulationShaping_BT_05),
  //CCA threshold
  //Must be 10 dB above receiver sensitivity multiplied by 2
  //Where receiver sensitivity S = S0 + 10log10(R/R0), R0 = 50kbps, R is the bit rate in kbps
    (173 >> 1)
  }
#endif
};

/************************************************************************************
*************************************************************************************
* Public memory declarations
*************************************************************************************
************************************************************************************/

phyPib_t gPhyPib;


/************************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
************************************************************************************/

/******************************************************************************
*Function name: PhyPib_InitOnStartUp
*
*Parameters passed in: none
*
*Return type: none
*
*Prototype: void PhyPib_InitOnStartUp(void)
*
*Description: This function performs the PHY PIB initialization: initialize 
*             the PIB structure with the default values, initialize 
*             the transceiver registers with the current PHY PIB parameters.
******************************************************************************/


void PhyPib_InitOnStartUp(void)
{
  phyMode_t *pPhyModeSupported = (phyMode_t *) &gPhyPib.mPIBphyModeSupported;
  phyMode_t *pPhyCurrentMode = (phyMode_t *) &gPhyPib.mPIBphyCurrentMode;
    
/******************************************************************
** gPhyPibPhyModeSupported_c bitmap defintion                     |
** Bit pos:|    15-9      |     8-4        |          3-0         |
**         |  Reserved    | Frequency band |  Bitmap for PHY mode |
*******************************************************************/
  
  //ST  -- will be initialized in MAC PIB
  gPhyPib.mPIBphyFSKScramblePSDU = TRUE;
  gPhyPib.mPIBphyTransmitPower = 0x1F;
  gPhyPib.mPIBphyFSKPreambleRepetitions = 16;
  gPhyPib.mPIBphyCurrentChannel = 0;
  //END
  
  #if gFreqBand_902__928MHz_d == 1
	  mFreqBand = g902__928MHz_c;
  #endif
  #if gFreqBand_950__958MHz_d == 1
	  mFreqBand = g950__958MHz_c;
  #endif
  #if gFreqBand_863__870MHz_d == 1
	  mFreqBand = g863__870MHz_c;
  #endif
  
  switch(mFreqBand)
  {
    case g863__870MHz_c:
      pPhyCurrentMode->bitAccess.freqBand = g863__870MHz_c;
      
      pPhyModeSupported->bitAccess.phyMode = (gPhyMode0_c | gPhyMode1_c);
      pPhyCurrentMode->bitAccess.phyMode = (gPhyMode0_c);
      
      gPhyPib.pPIBphyRfConstants = (phyRFConstants_t *) &phyPibRFConstants[0];
    break;
    case g950__958MHz_c:
      pPhyCurrentMode->bitAccess.freqBand = g950__958MHz_c;
      
      pPhyModeSupported->bitAccess.phyMode = (gPhyMode0_c | gPhyMode2_c);
      pPhyCurrentMode->bitAccess.phyMode = (gPhyMode0_c);
      
      gPhyPib.pPIBphyRfConstants = (phyRFConstants_t *) &phyPibRFConstants[0];
    break;
    default:
      pPhyModeSupported->bitAccess.freqBand = 
      pPhyCurrentMode->bitAccess.freqBand = g902__928MHz_c;
      
      pPhyModeSupported->bitAccess.phyMode = (gPhyMode0_c | gPhyMode1_c | gPhyMode2_c );
      pPhyCurrentMode->bitAccess.phyMode = (gPhyMode0_c);
      gPhyPib.pPIBphyRfConstants = (phyRFConstants_t *) &phyPibRFConstants[0];
    break;
  }

  PhyPib_RFUpdateModulationParameters();
  PhyPib_RFUpdateRFfrequency();
  PhyPib_RFUpdateDCfreeEncoding();
  PhyPib_RFUpdatePreambleLength();
  PhyPib_RFUpdatePowerStep();                                           
}

/******************************************************************************
*Function name: PhyPib_Get
*
*Parameters passed in: uint8_t attribute, void * pAttributeValue
*
*Return type: uint8_t
*
*Prototype: uint8_t PhyPib_Get(uint8_t attribute, void * pAttributeValue)
*
*Description: This function is used to get the value of the attribute. This function 
*             performs the following action: checks the input parameters, return
*             an error exit code if the wrong parameters are supplied, 
*             otherwise placed the value of the attribute at the location *pAttributeValue
******************************************************************************/

uint8_t PhyPib_Get(uint8_t attribute, void * pAttributeValue)
{
  uint8_t status = gPhyPibSuccess_c;
  if(pAttributeValue == NULL)
  {
    return gPhyPibInvalidParameter_c;
  }
  switch(attribute)
  {
    case gPhyPibPhyModeSupported_c:
      *(uint16_t *)pAttributeValue = gPhyPib.mPIBphyModeSupported;
    break;
    case gPhyPibCurrentMode_c:
      *(uint16_t *)pAttributeValue = gPhyPib.mPIBphyCurrentMode;
    break;
    case gPhyPibCurrentChannel_c:
      *(uint8_t *)pAttributeValue = gPhyPib.mPIBphyCurrentChannel;
    break;
    case gPhyPibTransmitPower_c:
      *(uint8_t *)pAttributeValue = gPhyPib.mPIBphyTransmitPower;
    break;
    case gPhyPibFSKPreambleRepetitions_c:
      *(uint16_t *)pAttributeValue = gPhyPib.mPIBphyFSKPreambleRepetitions;
    break;
    case gPhyPibFSKScramblePSDU_c:
      *(uint8_t *)pAttributeValue = gPhyPib.mPIBphyFSKScramblePSDU;
    break;
    default:
      status = gPhyPibUnsupportedAttribute;
    break;
  }
  return status;
}

/******************************************************************************
*Function name: PhyPib_SetCurrentPhyMode
*
*Parameters passed in: uint16_t phyMode
*
*Return type: uint8_t
*
*Prototype: uint8_t PhyPib_SetCurrentPhyMode(uint16_t phyMode)
*
*Description: This function is used to set the current PHY operating mode. 
*             This function performs the following action: checks the phyMode 
*             input parameter, return an error exit code if the wrong parameter 
*             is supplied, otherwise update the PIB structure and Radio registers 
*             based on phyMode value.
******************************************************************************/

uint8_t PhyPib_SetCurrentPhyMode(uint16_t phyMode)
{
  phyMode_t *pPhyModeSupported = (phyMode_t *) &gPhyPib.mPIBphyModeSupported;
  phyMode_t *pPhyCurrentMode = (phyMode_t *) &phyMode;
  
  if( gIdle_c != PhyGetState() )
  {
    return gPhyPibBusyTransceiver_c;
  }
  
  if(pPhyCurrentMode->bitAccess.freqBand != pPhyModeSupported->bitAccess.freqBand)
  {
    return gPhyPibInvalidParameter_c;
  }
  
  if((pPhyModeSupported->bitAccess.phyMode & pPhyCurrentMode->bitAccess.phyMode ) != pPhyCurrentMode->bitAccess.phyMode)
  {
    return gPhyPibInvalidParameter_c;
  }
  
  if( !( (pPhyCurrentMode->bitAccess.phyMode == gPhyMode0_c) ||
         (pPhyCurrentMode->bitAccess.phyMode == gPhyMode1_c) ||
         (pPhyCurrentMode->bitAccess.phyMode == gPhyMode2_c) ))
  {
    
    return gPhyPibInvalidParameter_c;
  }
  
  if((pPhyCurrentMode->bitAccess.freqBand == g863__870MHz_c) || (pPhyCurrentMode->bitAccess.freqBand == g950__958MHz_c))
  {
    
    if(pPhyCurrentMode->bitAccess.phyMode == gPhyMode0_c)
    {
      gPhyPib.pPIBphyRfConstants = (phyRFConstants_t *) &phyPibRFConstants[0]; 
    }
    
    else
    {
      gPhyPib.pPIBphyRfConstants = (phyRFConstants_t *) &phyPibRFConstants[1];
    }
  }
  else
  {
  
    if(pPhyCurrentMode->bitAccess.phyMode == gPhyMode0_c)
    {
      gPhyPib.pPIBphyRfConstants = (phyRFConstants_t *) &phyPibRFConstants[0]; 
    }
    
    else if(pPhyCurrentMode->bitAccess.phyMode == gPhyMode1_c)
    {
      gPhyPib.pPIBphyRfConstants = (phyRFConstants_t *) &phyPibRFConstants[1];
    }
    else
    {
      gPhyPib.pPIBphyRfConstants = (phyRFConstants_t *) &phyPibRFConstants[2];
    }
  }

  gPhyPib.mPIBphyCurrentMode = phyMode;
  gPhyPib.mPIBphyCurrentChannel = 0;
  
  PhyPib_RFUpdateModulationParameters();
  PhyPib_RFUpdateRFfrequency();
  
  return gPhyPibSuccess_c;
}

/******************************************************************************
*Function name: PhyPib_SetCurrentChannel
*
*Parameters passed in: uint8_t channel
*
*Return type: uint8_t
*
*Prototype: uint8_t PhyPib_SetCurrentChannel(uint8_t channel)
*
*Description:  This function is used to set the current logical channel. 
*              The channel will be used for every transceiver operation 
*              that requires RF: CCA, ED, RX and TX. This function performs 
*              the following action: checks the channel input parameter, 
*              return an error exit code if the wrong parameter is supplied, 
*              otherwise update the PIB structure and Radio registers based 
*              on channel value.
******************************************************************************/

uint8_t PhyPib_SetCurrentChannel(uint8_t channel)
{
  
  if( gIdle_c != PhyGetState() )
  {
    return gPhyPibBusyTransceiver_c;
  }
  
  if(channel >= gPhyPib.pPIBphyRfConstants->totalNumChannels) 
  {
    return gPhyPibInvalidParameter_c;
  }

  gPhyPib.mPIBphyCurrentChannel = channel;
  
  PhyPib_RFUpdateRFfrequency();

  return gPhyPibSuccess_c;

}

/******************************************************************************
*Function name: PhyPib_SetCurrentTransmitPower
*
*Parameters passed in: uint8_t pwrLevel
*
*Return type: uint8_t
*
*Prototype: uint8_t PhyPib_SetCurrentTransmitPower(uint8_t pwrLevel)
*
*Description: This function is used to set the current power step. 
*             The power step will be used for every transceiver operation that requires 
*             RF: TX. This function performs the following action: checks the pwrLevel 
*             input parameter, return an error exit code if the wrong parameter is supplied, 
*             otherwise update the PIB structure and Radio registers based on pwrLevel value.
******************************************************************************/

uint8_t PhyPib_SetCurrentTransmitPower(uint8_t pwrLevel)
{
  
  if( gIdle_c != PhyGetState() )
  {
    return gPhyPibBusyTransceiver_c;
  }
  if( pwrLevel > 0x1F )
  {
    return gPhyPibInvalidParameter_c;
  }
  
  gPhyPib.mPIBphyTransmitPower = pwrLevel;
  
  PhyPib_RFUpdatePowerStep();                 
  
  return gPhyPibSuccess_c;

}

/******************************************************************************
*Function name: PhyPib_SetCurrentFSKPreambleRepetitions
*
*Parameters passed in: uint16_t preambleLength
*
*Return type: uint8_t
*
*Prototype: uint8_t PhyPib_SetCurrentFSKPreambleRepetitions(uint16_t preambleLength)
*
*Description: This function is used to set the current preamble length. 
*             The preamble length will be used for every transceiver operation that requires 
*             RF: TX and RX. This function performs the following action: checks the preambleLength
*             input parameter, return an error exit code if the wrong parameter is supplied, 
*             otherwise update the PIB structure and Radio registers based on preambleLength value.
******************************************************************************/
uint8_t PhyPib_SetCurrentFSKPreambleRepetitions(uint16_t preambleLength)
{
  if( gIdle_c != PhyGetState() )
  {
    return gPhyPibBusyTransceiver_c;
  }
  if((preambleLength > gPhyPibMaxPmbLen_c) || (preambleLength < gPhyPibMinPmbLen_c))
  {
    return gPhyPibInvalidParameter_c;
  }
  gPhyPib.mPIBphyFSKPreambleRepetitions = preambleLength;
  
  PhyPib_RFUpdatePreambleLength();

  return gPhyPibSuccess_c;
  
}

/******************************************************************************
*Function name: PhyPib_SetCurrentFSCScrablePSDU
*
*Parameters passed in: bool_t scrablePSDU
*
*Return type: uint8_t
*
*Prototype: uint8_t PhyPib_SetCurrentFSCScrablePSDU(bool_t scrablePSDU)
*
*Description: This function is used to enable/disable the Radio DC-free 
*             encoding/decoding mechanism (Data Whitening). 
******************************************************************************/

uint8_t PhyPib_SetCurrentFSCScrablePSDU(bool_t scrablePSDU)
{
  if( gIdle_c != PhyGetState() )
  {
    return gPhyPibBusyTransceiver_c;
  }
  gPhyPib.mPIBphyFSKScramblePSDU = scrablePSDU;
  
  PhyPib_RFUpdateDCfreeEncoding();
  
  return gPhyPibSuccess_c;
}

/******************************************************************************
*Function name: PhyPib_Set
*
*Parameters passed in: uint8_t attribute, void * pAttributeValue
*
*Return type: uint8_t
*
*Prototype: uint8_t PhyPib_Set(uint8_t attribute, void * pAttributeValue)
*
*Description: This function is used to set the attribute with the value pointed by pAttributeValue.
*             This function performs the following action: checks the input parameters, 
*             return an error exit code if the wrong parameters are supplied, 
*             otherwise update the PIB structure and Radio registers 
*             based on attribute and *pAttributeValue value.
******************************************************************************/

uint8_t PhyPib_Set(uint8_t attribute, void * pAttributeValue)
{
  uint8_t status = gPhyPibSuccess_c;
  
  if(pAttributeValue == NULL)
  {
    return gPhyPibInvalidParameter_c;
  }
  
  switch(attribute)
  {
    case gPhyPibPhyModeSupported_c:
      status = gPhyPibReadOnly_c;
    break;
    case gPhyPibCurrentMode_c:
      status = PhyPib_SetCurrentPhyMode(*(uint16_t *) pAttributeValue);
    break;
    case gPhyPibCurrentChannel_c:
      status = PhyPib_SetCurrentChannel(*(uint8_t *) pAttributeValue);
    break;
    case gPhyPibTransmitPower_c:
      status = PhyPib_SetCurrentTransmitPower(*(uint8_t *) pAttributeValue);
    break;
    case gPhyPibFSKPreambleRepetitions_c:
      status = PhyPib_SetCurrentFSKPreambleRepetitions(*(uint16_t *) pAttributeValue);
    break;
    case gPhyPibFSKScramblePSDU_c:
      status = PhyPib_SetCurrentFSCScrablePSDU(*(uint8_t *) pAttributeValue);
    break;
    default:
      status = gPhyPibUnsupportedAttribute;
    break;
  }
  return status;
}

/************************************************************************************
*************************************************************************************
* Private functions
*************************************************************************************
************************************************************************************/

/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/

void PhyPib_RFUpdateDCfreeEncoding(void)
{
  uint8_t packetConfig1Reg;
  packetConfig1Reg = MKW01Drv_ReadRegister(MKW01_Reg_PacketConfig1);
  
  /* disable DC-free encoding/decoding */
  packetConfig1Reg &= 0x9F;
  
  if(gPhyPib.mPIBphyFSKScramblePSDU == TRUE)
  {
    /* enable DC-free encoding/decoding (Whitening) */
    packetConfig1Reg |= 0x40;
  }

  /* Radio packet mode config @0x37*/
    
  MKW01Drv_WriteRegister(MKW01_Reg_PacketConfig1, (uint8_t) packetConfig1Reg);
}

/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/

void PhyPib_RFUpdatePreambleLength(void)
{
  /* Radio preamble size initialization @0x2C-0x2D*/
  
  MKW01Drv_WriteRegister(MKW01_Reg_PreambleMsb, (uint8_t) (gPhyPib.mPIBphyFSKPreambleRepetitions >> 8));
  MKW01Drv_WriteRegister(MKW01_Reg_PreambleLsb, (uint8_t) (gPhyPib.mPIBphyFSKPreambleRepetitions));
}

/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/

void PhyPib_RFUpdatePowerStep(void)
{
  uint8_t pwrReg;
  pwrReg = MKW01Drv_ReadRegister(MKW01_Reg_PaLevel);
  pwrReg &= 0xE0;
  pwrReg |= (gPhyPib.mPIBphyTransmitPower & 0x1F);
  
  /* Radio output power initialization @0x11*/
  
  MKW01Drv_WriteRegister(MKW01_Reg_PaLevel, (uint8_t) (pwrReg) );
}

/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/

void PhyPib_RFUpdateRFfrequency(void)
{
  uint32_t channelFreq;
  uint8_t  channelFreqArray[4] = {0,0,0,0};
  
  phyMode_t *pPhyCurrentMode = (phyMode_t *) &gPhyPib.mPIBphyCurrentMode;
  
  switch(pPhyCurrentMode->bitAccess.freqBand)
  {
    case g863__870MHz_c:
    {
      if(pPhyCurrentMode->bitAccess.phyMode == gPhyMode0_c)
      {
        *((uint16_t *)&channelFreqArray[0]) = (uint16_t) gPhyPib.mPIBphyCurrentChannel;
        channelFreq = *((uint32_t *) &channelFreqArray[0]) >> 1;
        channelFreq = channelFreq / 10;
        channelFreq += 14141440;
      }
      else if(pPhyCurrentMode->bitAccess.phyMode == gPhyMode1_c)
      {
        *((uint16_t *)&channelFreqArray[0]) = (uint16_t) (((uint16_t )gPhyPib.mPIBphyCurrentChannel * 4) + 1);
        channelFreq = *((uint32_t *) &channelFreqArray[0]) >> 2;
        channelFreq = channelFreq / 10;
        channelFreq += 14141440;
      }
    }
    break;
    case g950__958MHz_c:
    {
      if(pPhyCurrentMode->bitAccess.phyMode == gPhyMode0_c)
      {
        *((uint16_t *)&channelFreqArray[0]) = (uint16_t) gPhyPib.mPIBphyCurrentChannel;
        channelFreq = *((uint32_t *) &channelFreqArray[0]) >> 1;
        channelFreq = channelFreq / 10;
        channelFreq += 15581184;
      }
      else  if(pPhyCurrentMode->bitAccess.phyMode == gPhyMode2_c)
      {
        *((uint16_t *)&channelFreqArray[0]) = (uint16_t) (((uint16_t )gPhyPib.mPIBphyCurrentChannel * 3) + 1);
        channelFreq = *((uint32_t *) &channelFreqArray[0]) >> 1;
        channelFreq = channelFreq / 10;
        channelFreq += 15581184;
      }
    }
    break;
    default:
    {
      if(pPhyCurrentMode->bitAccess.phyMode == gPhyMode0_c)
      {
        
        *((uint16_t *)&channelFreqArray[0]) = (uint16_t) gPhyPib.mPIBphyCurrentChannel;
        channelFreq = *((uint32_t *) &channelFreqArray[0]) >> 2;
        channelFreq += 14786560;

      }
      else if(pPhyCurrentMode->bitAccess.phyMode == gPhyMode1_c)
      {
        *((uint16_t *)&channelFreqArray[0]) = (uint16_t) gPhyPib.mPIBphyCurrentChannel;
        channelFreq = *((uint32_t *) &channelFreqArray[0]) >> 1;
        channelFreq += 14802944;
      }
      else if(pPhyCurrentMode->bitAccess.phyMode == gPhyMode2_c)
      {
        *((uint16_t *)&channelFreqArray[0]) = (uint16_t) gPhyPib.mPIBphyCurrentChannel;
        channelFreq = *((uint32_t *) &channelFreqArray[0]) >> 1;
        channelFreq += 14802944;
      }
    }
    break;
  }

  /* Radio RF frequency initialization @0x07-0x09*/
    
  MKW01Drv_WriteRegister(MKW01_Reg_FrfMsb, (uint8_t) ( channelFreq >> 16 ) );
  MKW01Drv_WriteRegister(MKW01_Reg_FrfMid, (uint8_t) ( channelFreq >> 8 ) );
  MKW01Drv_WriteRegister(MKW01_Reg_FrfLsb, (uint8_t) ( channelFreq ));
}

/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/

void PhyPib_RFUpdateModulationParameters(void)
{
  /* Radio bit rate initialization @0x03-0x04*/
   
  MKW01Drv_WriteRegister(MKW01_Reg_BitrateMsb, (uint8_t) ( gPhyPib.pPIBphyRfConstants->bitRateReg >> 8 ) );
  MKW01Drv_WriteRegister(MKW01_Reg_BitrateLsb, (uint8_t) ( gPhyPib.pPIBphyRfConstants->bitRateReg & 0x00FF) );
  
  /* Radio frequency deviation initialization @0x05-0x06*/
  
  MKW01Drv_WriteRegister(MKW01_Reg_FdevMsb, (uint8_t) ( gPhyPib.pPIBphyRfConstants->fdevReg >> 8 ) );
  MKW01Drv_WriteRegister(MKW01_Reg_FdevLsb, (uint8_t) ( gPhyPib.pPIBphyRfConstants->fdevReg & 0x00FF) );
  
  /* Radio channel filter bandwidth initialization @0x19*/
  
  MKW01Drv_WriteRegister(MKW01_Reg_RxBw, gPhyPib.pPIBphyRfConstants->rxBwReg);
  
  /* Radio channel filter bandwidth for AFC operation initialization @0x1A*/
  
  MKW01Drv_WriteRegister(MKW01_Reg_AfcBw, gPhyPib.pPIBphyRfConstants->rxBwAfcReg);
}



